/* timing~.c  one float inlet, one signal outlet
** 
** 02/11/04 IF
** 04/10/26 IF 
*/

#include <math.h>
#include "ext.h"    // Required for all Max external objects
#include "z_dsp.h"  // Required for all MSP external objects

void *oscil_class;

#define TABSIZE 1024
#define DEF_FREQ 440

typedef struct _oscil 	// Data structure for this object
{
    t_pxobject t_obj;
    t_float t_freq;  // current frequenct
    t_float t_sr;	// smapling rate
    t_float t_ptr;	// current phase
    void *t_outlet; // float out
    t_float t_tab[TABSIZE + 1];
    t_float t_out[1001];
} t_oscil;

void *oscil_new(float freq);
t_int *oscil_perform(t_int *w);
void oscil_float(t_oscil *x, t_float val);
void oscil_bang(t_oscil *x);
void oscil_sin(t_oscil *x);
void oscil_interp(t_oscil *x);
void oscil_filltable(t_oscil *x);
void oscil_trunc(t_oscil *x);

int main(void)
{
    setup((t_messlist **)&oscil_class, (method)oscil_new, 0L, (short)sizeof(t_oscil), 0L, A_DEFFLOAT, 0);
    addfloat((method)oscil_float);
    addbang((method)oscil_bang);
	return(0);
}

void *oscil_new(float freq)
{
	t_oscil *x = (t_oscil *)newobject(oscil_class);
	if (freq != 0)
		x->t_freq = freq;
	else
		x->t_freq = DEF_FREQ;
	
	x->t_ptr = 0;
	x->t_sr = sys_getsr();
	x->t_outlet = floatout(x);
	oscil_filltable(x);

	return (x);
}

void oscil_float(t_oscil *x, t_float val)
{
	x->t_freq = val;
}

void oscil_filltable(t_oscil *x)
{
	int i;
	for (i = 0; i < TABSIZE; i++)
		x->t_tab[i] = sin(2 * PI * i / TABSIZE);
	x->t_tab[TABSIZE] = x->t_tab[0];
}


void oscil_bang(t_oscil *x)
{
	oscil_sin(x);
	oscil_trunc(x);
	oscil_interp(x);
}

void oscil_sin(t_oscil *x)
{

	double beg, end;
	int i, n;
	float p, incr = 2 * PI * x->t_freq / x->t_sr;
	
	clock_getftime(&beg);
	p = x->t_ptr; // starting phase
	
	for (i = 0; i < 1000; i++)
	{	
		float *out = x->t_out;
		n = 1000;
		
		while (n--) 
		{
			*out++ = sin(p);
			p += incr;
			if (p > x->t_sr)
				p -= x->t_sr;
		}
	}
	x->t_ptr = p; // ending phase
	clock_getftime(&end);

post("SIN: %5.3f msec per 1000",(end - beg) / 1000);


}

void oscil_interp(t_oscil *x)
{
	double beg, end;
	int i, n, ip;
	float p, incr = TABSIZE * x->t_freq / x->t_sr;
	p = x->t_ptr; // starting phase
	
	clock_getftime(&beg);
	for (i = 0; i < 1000; i++)
	{	
		float *out = x->t_out;
		n = 1000;
		
	p = x->t_ptr;
		while (n--) 
		{
			ip = (int)p;
			*out++ = (x->t_tab[ip + 1] - x->t_tab[ip]) * (p - ip) + x->t_tab[ip];
			p += incr;
			if (p > TABSIZE)
				p -= TABSIZE;
		}
	}
	x->t_ptr = p; // ending phase
	clock_getftime(&end);

post("INTERP: %5.3f msec per 1000", (end - beg) / 1000);


}

void oscil_trunc(t_oscil *x)
{
	double beg, end;
	int i, n;
	float p, incr = TABSIZE * x->t_freq / x->t_sr;
	p = x->t_ptr; // starting phase
	
	clock_getftime(&beg);
	for (i = 0; i < 1000; i++)
	{	
		float *out = x->t_out;
		n = 1000;
		
	p = x->t_ptr;
		while (n--) 
		{
	
			*out++ = x->t_tab[(int)(p + 0.5)];
			p += incr;
			if (p > TABSIZE)
				p -= TABSIZE;
		}
	}
	x->t_ptr = p; // ending phase
	clock_getftime(&end);

post("TRUNC: %5.3f msec per 1000", (end - beg) / 1000);


}
/*
CodeWarrior data on PB 1.25GHz
SIN: 0.201000 msec per 1000
TRUNC: 0.092000 msec per 1000
INTERP: 0.182245 msec per 1000
SIN: 0.423814 msec per 1000
TRUNC: 0.088000 msec per 1000
INTERP: 0.189000 msec per 1000
SIN: 0.442289 msec per 1000
TRUNC: 0.077000 msec per 1000
INTERP: 0.255000 msec per 1000
SIN: 0.678037 msec per 1000
TRUNC: 0.109000 msec per 1000
INTERP: 0.251000 msec per 1000
SIN: 0.330000 msec per 1000
TRUNC: 0.072000 msec per 1000
INTERP: 0.212000 msec per 1000
SIN: 0.323660 msec per 1000
TRUNC: 0.143000 msec per 1000
INTERP: 0.173000 msec per 1000
SIN: 0.325000 msec per 1000
TRUNC: 0.074000 msec per 1000
INTERP: 0.245000 msec per 1000
SIN: 0.474173 msec per 1000
TRUNC: 0.120000 msec per 1000
INTERP: 0.217000 msec per 1000
 */
